package com.hitachi.couriba.web.security;

import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.annotation.Order;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.config.annotation.web.configurers.AbstractHttpConfigurer;
import org.springframework.security.config.http.SessionCreationPolicy;
import org.springframework.security.web.SecurityFilterChain;
import org.springframework.security.web.authentication.UsernamePasswordAuthenticationFilter;

/**
 * Spring Securityの設定を定義し、APIキー認証フィルターを統合する設定クラス.
 * */
@Configuration
@EnableWebSecurity
public class SecurityConfig {

  @Value("${application.security.app-api-key}")
  private String appApiKey;

  @Value("${application.security.api-key}")
  private String serverApiKey;

  @Bean
  public AuthenticationFilter appAuthenticationFilter() {
    SecurityAuthenticationService authenticationService = new SecurityAuthenticationService(
        appApiKey, "{\"error\":\"Invalid API Key\"}"
    );
    return new AuthenticationFilter(authenticationService);
  }

  @Bean
  public AuthenticationFilter serverAuthenticationFilter() {
    SecurityAuthenticationService authenticationService = new SecurityAuthenticationService(
        serverApiKey, "{\"code\":1,\"message\":\"Invalid API Key\"}"
    );
    return new AuthenticationFilter(authenticationService);
  }

  @Bean
  @Order(1)
  public SecurityFilterChain appFilterChain(HttpSecurity http, AuthenticationFilter appAuthenticationFilter) throws Exception {
    http.securityMatcher("/display/authentication/**")
        .csrf(AbstractHttpConfigurer::disable)
        .authorizeHttpRequests(authorizationManagerRequestMatcherRegistry ->
            authorizationManagerRequestMatcherRegistry.anyRequest().authenticated())
        .sessionManagement(httpSecuritySessionManagementConfigurer ->
            httpSecuritySessionManagementConfigurer.sessionCreationPolicy(SessionCreationPolicy.STATELESS))
        .addFilterBefore(appAuthenticationFilter, UsernamePasswordAuthenticationFilter.class);
    return http.build();
  }

  @Bean
  @Order(2)
  public SecurityFilterChain serverFilterChain(HttpSecurity http, AuthenticationFilter serverAuthenticationFilter) throws Exception {
    http.securityMatcher("/display/actuator", "/display/entrySwitch", "/display/endPayment")
        .csrf(AbstractHttpConfigurer::disable)
        .authorizeHttpRequests(authorizationManagerRequestMatcherRegistry ->
            authorizationManagerRequestMatcherRegistry
                .requestMatchers("/display/actuator").permitAll()
                .requestMatchers("/display/entrySwitch", "/display/endPayment").authenticated()
                .anyRequest().denyAll())
        .sessionManagement(httpSecuritySessionManagementConfigurer ->
            httpSecuritySessionManagementConfigurer.sessionCreationPolicy(SessionCreationPolicy.STATELESS))
        .addFilterBefore(serverAuthenticationFilter, UsernamePasswordAuthenticationFilter.class);
    return http.build();
  }
}